package gov.va.med.mhv.integration.phr.transfer;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonIgnore;

/**
 * User: Rob Murtha
 */
public class PatientExtractStatus implements Serializable {

    private static final long serialVersionUID = 5817436997570007852L;
    /** patient identifier */
    private PatientIdentifier patient;

    /** date when this status was retrieved */
    private Date retrievedDate;

    /** date when this status was retrieved */
    private Date lastRefreshDate;

    static final String DATEONLY_FORMAT_STRING = "dd MMM yyyy";
    
    /** list of status for each facility and domain */
    private List<FacilityExtractStatus> facilityExtractStatusList = new ArrayList<FacilityExtractStatus>();


    public PatientExtractStatus(){}

    public PatientExtractStatus(PatientIdentifier patient) {
        this.patient=patient;
        this.retrievedDate = new Date();
    }

    /** Set the last requested date using a refreshDate instead of a queryDate */
    public void setLastRefreshDate(Date lastRefreshDate) {
        this.lastRefreshDate=lastRefreshDate;
    }

    public Date getLastRefreshDate() {
        return lastRefreshDate;
    }


    public boolean needsRefresh() {
        boolean is = false;
        if(null==lastRefreshDate || !isSameDate(lastRefreshDate, new Date()))
            is = true;

        return is;
    }

    /**
     * Overall status of refresh requests.
     * Provides a rolled up value of true if all requests have been processed and completed.
     * @return  false if any requests are outstanding
     */
    @JsonIgnore
    public boolean isCompleted() {
        boolean is = false;

        if(needsRefresh())
            is = false;
        else {
            if(facilityExtractStatusList!=null) {
                Date today = new Date();
                for (FacilityExtractStatus facilityExtractStatus : facilityExtractStatusList) {
                    is=isSameDate(today,facilityExtractStatus.getLastRequested()) && facilityExtractStatus.isUpToDate();
                    if(!is) {
                        break;
                    }
                }
            }
        }
        return is;
    }

    @JsonIgnore
    public Set<String> getInProgressExtracts() {
        Set<String> result = new HashSet<String>();

        Date today = new Date();

        // add comparison to check if the individual refresh request has been processed today
        // this is necessary to ensure that the status is inProgress while waiting for the refresh to start
        // condition for completion is that the request is made today and that the completion status is true

        for (FacilityExtractStatus fes : facilityExtractStatusList) {
        	
        	/*if(fes.getLastCompleted() != null)
        	{
        		 long lastRequested = (fes.getLastRequested() == null) ? 0L : fes.getLastRequested().getTime();
        		 long lastCompleted = (fes.getLastCompleted() == null) ? 0L : fes.getLastCompleted().getTime();
        		 
        		 fes.isCompleted()
        		 
        		 if(lastCompleted < lastRequested)
        		 {
        			 result.add(fes.getExtract());
        		 }
        	}*/
        	
        	if(fes.inProgress())
        	{
        		result.add(fes.getExtract());
        	}
        	
        	
//            if(!(isSameDate(today,fes.getLastRequested()) && fes.isCompleted())) {
//                result.add(fes.getExtract());
//            }
        }
        return result;
    }

    public PatientIdentifier getPatient() {
        return patient;
    }

    public void setPatient(PatientIdentifier patient) {
        this.patient = patient;
    }

    public Date getRetrievedDate() {
        return retrievedDate;
    }

    public void setRetrievedDate(Date retrievedDate) {
        this.retrievedDate = retrievedDate;
    }

    public List<FacilityExtractStatus> getFacilityExtractStatusList() {
        return facilityExtractStatusList;
    }

    public void setFacilityExtractStatus(List<FacilityExtractStatus> facilityExtractStatusList) {
        this.facilityExtractStatusList = facilityExtractStatusList;
    }

    public boolean isSameDate(Date date1, Date date2) {
        boolean result = false;
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(DATEONLY_FORMAT_STRING);
            String temp1 = sdf.format(date1);
            String temp2 = sdf.format(date2);
            result = temp1.equals(temp2);
        } catch (Exception ignore) {
            //System.out.println(ignore.getMessage());
        }
        return result;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        PatientExtractStatus that = (PatientExtractStatus) o;

        if (facilityExtractStatusList != null ? !facilityExtractStatusList.equals(that.facilityExtractStatusList) : that.facilityExtractStatusList != null)
            return false;
        if (patient != null ? !patient.equals(that.patient) : that.patient != null) return false;
        if (retrievedDate != null ? !retrievedDate.equals(that.retrievedDate) : that.retrievedDate != null)
            return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result;
        result = (patient != null ? patient.hashCode() : 0);
        result = 31 * result + (retrievedDate != null ? retrievedDate.hashCode() : 0);
        result = 31 * result + (facilityExtractStatusList != null ? facilityExtractStatusList.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return "PatientExtractStatus{" +
                "patient=" + patient +
                ", retrievedDate=" + retrievedDate +
                ", lastRefreshDate=" + lastRefreshDate +
                ", facilityExtractStatusList=" + facilityExtractStatusList +
                '}';
    }
}
